; File TIME.S includes the SIO2PC routines related to using
; and programming the timer chip, and timing routines in general
;
;
;  Includes:

;       ACC_DLY;  TIMER_0;  TIME_LAG;   PRG_T0
;       MYVEC8;  FIXV8;  COLSEV8;  ONE_TIK;  TIMING
;       SETUPV8;  DO_TIME; HOWFAST

; The following notes copied from NEW_TIME.ASM

; Notes: The 8254 chip is clocked at 1.193 Mhz. The period is
; 840 nanoseconds. Timer 2 is gated by bit 0 of port 61h. Timer 2
; produces the sound through the speaker. Bit 1 of port 61h gates
; the timer's output to the speaker (1 = enable sound).


; More info on what port 61 bits 0 and 1 REALLY do:
;
; First, the 8254 timer chip has a Gate pin for each counter.
; The counter generally won't do anything unless that pin is high.
; (On counters 0 and 1, the Gate is tied high.)
;
; Bit 0 of PORT 061h goes to counter 2's gate pin.  This means that
; if you want that timer to do ANYTHING, produce sound, count, whatever,
; you must set bit 0 to high.
; OK.  The output of counter two goes to an AND gate which drives
; the speaker.  The second input of the AND gate is bit 1 of PORT 061h.
; Obviously, either input to the AND gate can block output to the speaker
; by being LOW.  And either input can control the speaker by itself if
; the other input is HIGH.

; YOU are controlling the input using Port 061h, bit 1.  The timer is
; controlling the other input.  In most cases, the timer's OUT pin is
; resting in the HIGH state.  But depending on the MODE, it may be low
; at least part of the time.

; Gate = 0 makes OUT high in modes 2 & 3 and has no effect on OUT in the
; other modes
; Gate = 0 disables counting in modes 0, 2, 3, & 4
; OUT is low during counting, then stays high in modes 0 and 1
; Out is low for 1 clock after the countdown in modes 2, 4, & 5, high
; otherwise.
; Out is low and high with a 50% duty cycle in mode 3 while counting
; Modes 1 and 5 are triggered by a low to high gate transition.
;
; If you want to control the speaker directly, put the 8254's OUT2
; pin into a high state and use Port 061h, bit 1.
; If you want the 8254 to drive the speaker, set Port 061H, bit 1
; to logic 1.
;

; Port #  dir:                  COMMENT

; 040h    R/W   Counter 0 data. Counter 0 is sys. timer; IRQ 0
; 041h    R/W   Counter 1 data. Used for refresh, 15 usec. period
; 042h    R/W   Counter 2 data. Gated by 61h, b0, used for speaker.
; 043h     W    Control word.

COMMENT\

 CONTROL WORD FORMAT:  (PORT 043h)

   BIT 7  BIT 6                                                    VALUE
    1       1           Read back command, see below                0Ch
    0       1           Select counter 1                            40h
    1       0           Select counter 2                            80h
    0       0           Select counter 0                            00H

   BIT 5  BIT 4
    0       0           Counter latch command                       00h
    0       1           Read/Write LSB only                         10h
    1       0           Read/Write MSB only                         20h
    1       1           Read/Write LSB, then MSB                    30h

Note: the remaining bits are don't care for Counter Latch command, and
have different meanings (see below) for Read Back command. Otherwise, they
are for mode programming as given below:

	BIT 3   BIT 2   BIT 1                                  VALUE
	 0       0       0      MODE 0                          00h
	 0       0       1      MODE 1                          02h
	 X       1       0      MODE 2                          04h
	 X       1       1      MODE 3                          06h
	 1       0       0      MODE 4                          08h
	 1       0       1      MODE 5                          09h

 BIT 0 is 0 for binary counter (normal) and 1 for BCD counter.

Direct speaker control via PORT 061h:

bit 0: set (1) for normal control by timer 2
bit 0: clear (0) for direct control
bit 1: set to turn speaker on (cone out) when bit 0 is 0
bit 1: clear to turn speaker off (cone in) when bit 0 is 0

Ref. Using Assembly Language page 386

ENDOFCOMMENT\

; A routine to waste just a few machine cycles, mainly for
; slow ports, between successive IN's and OUT's.

TIME_LAG:
        NOP
        NOP
        NOP
        NOP
        JMP >L1
L1:     RET


; A routine to delay until VARBL has changed value (one tick)
; or 55 msec.
ONE_TIK:
        PUSH AX
        MOV AX,VARBL
        CMP VARBL,0; It won't change if it's zero
        JNE >L1
        POP AX
        RET
L1:     CMP AX,VARBL
        JNE  L1
        POP AX
        RET

; the above routine has a resolution of 1 tick.  The one below
; has more resolution:
; A post rev 3.14 addition

ONETICK:
	PUSH AX, CX
	MOV CX, 4
L1:	MOV BX, 08000h
	CALL TIMER_0
	LOOP L1
	POP CX, AX
	RET

;
;       A routine to redirect the timer 0 interrupt to my routine

; This routine demonstrates the following:

; 1) Getting an interrupt vector
; 2) Re-directing an interrupt

; This routine is useful as a timer to time a certain number of timer
; ticks, each being 55 ms. After re-setting the interrupt, the user
; puts the number of ticks desired into WORD VARBL. The interrupt
; routine decrements the word until it is 0, and stops decrementing.
; The user can just monitor VARBL until it becomes zero. The normal
; functions of the timer 0 routine (INT 08) are still enabled. Some
; counts are:

;       10 SECONDS:     182 COUNTS
;        5               91
;        3               55
;        2               36
;        1               18
;       .5                9
;       110 mS            2

; My interrupt routine:

MYVEC8:
        STI
        PUSH DS
        PUSH CS
        POP DS

; *** NOTE TECHNIQUE FOR "CALLING" ORIGINAL INT ROUTINE AS
;     THOUGH IT WERE A SUBROUTINE...  ****

        PUSHF; See Bible, pg 247; simulate interrupt...
        CALL FAR [CONTIN]; let old handler go first then IRET here
        CMP VARBL,0;    If it's already 0, don't change it
        JE >L1
        DEC VARBL
L1:     POP DS
        IRET

SETUPV8:

        MOV AL,08; Get INT 08 address
        MOV AH, 035h; Get interrupt vector function
        INT 021h; Now ES:BX holds interrupt vector
        MOV H_CONTIN,ES
        MOV CONTIN,BX

; Now change the interrupt vector to point to my routine:

        MOV AL,08; Set INT 08 address
        MOV AH,025h; Set interrupt DOS function
        MOV DX, OFFSET MYVEC8
        PUSH CS
        POP DS
        INT 021h; Change the interrupt
        RET

; Now, to close, I'll restore the original vector:

; Note below:, I had to load DX first, because DS is the ref.
; segment for the MOV instruction!

FIXV8:  
        MOV AH,025h
        MOV AL,08
        MOV DX, CONTIN
        MOV BX, H_CONTIN
        MOV DS,BX
        INT 021h

        PUSH CS; Get DS back to program segment
        POP DS; just because I like it that way...
        RET

; CLOSEV8 will be used to fix INT 08 vector from the INT 024h
; handler. It does the job without DOS because you're not supposed
; to call high DOS functions while in INT 024h handler. I'm relying
; on having interrupts turned off to keep me out of trouble.  Note:
; don't try to use TICKS macro after turning off this function!
; NOTE: if there are problems, turn off timer interrupts directly at
; chip.

CLOSEV8:
        CLI
        PUSH ES
        PUSH BX
        PUSH AX
        XOR BX,BX; Put 0 into BX, int table is in segment 0000
        MOV ES,BX
        MOV BX,020h; Location of INT 8 vector
        MOV AX, CONTIN
        MOV ES:[BX],AX
        MOV AX,H_CONTIN
        MOV ES:[BX+2],AX
        STI
        POP AX
        POP BX
        POP ES
        RET


; *******************************************************************

; Timing routines for timer #2 imported from NEW_TIME.ASM as part of
; rev. 4.08.  Intent is to use timer 2 when timer 0 is busy with
; VARBL related timing.

; *******************************************************************

COMMENT\

I'm taking this routine out with rev 4.21, 04/01/00

;       HOWFAST does a rough check on the speed of the computer
;       SIO2PC is currently running on

HOWFAST:
        PUSH AX, BX, CX, DX, BP
        MOV BP, 4 ; I'll take 4 readings and average them ...
        MOV VARBL, 2 ; I'll time for one tick, but throw the first
L1:     CMP VARBL, 1 ; away because it'll be a partial
        JNE L1
        XOR AX, AX
        XOR BX, BX
        XOR DX, DX
        XOR CX, CX

; counting loop:

L3:     INC AX          ; AX and BX will be my 32 bit loop counter
        JNZ > L2
        INC BX          ; INC BX when AX rolls over
L2:     CMP VARBL, 0    ; are we all done yet?
        JNE L3          ; if no, do some more incrementing

; end of counting loop

        MOV VARBL, 1    ; prepare for another count
        ADD CX, AX      ; CX holds low word, sum of counts
        ADC DX, BX      ; DX holds high word of sum
        DEC BP          ; BP counts the 4 trials
        JNZ L3          ; do another trial

; Now I have the sum of four trials in DX:CX

        SHR CX, 1       ; first part of divide by 4 is divide by two
        SHR DX, 1       ; for high word, I need to xfer carry to CX if
        JNC > L4        ; it exists
        OR CX, 08000h   ; sets high bit of CX
L4:     SHR CX, 1       ; 2nd divide by two
        SHR DX, 1
        JNC > L5
        OR CX, 08000h
L5:     MOV HFAST, DX   ; keep average count for ref
        MOV LFAST, CX   ; low word of count for ref

; Now, I'll compute how many counts to use for 0.2 us.  This will be
; the counts I've arrived at above divided by (0.055 * 5,000,000) or
; divided by 43238h, which is the same as D6D8h * 5, so I'll divide by
; those two quantities.  (Can't divide by > 16 bit number.)

B124:  MOV AX, CX       ; High word already in DX
       MOV BX, 0D6D8h   ; first, divide by this #
       DIV BX           ; divides DX:AX by BX, quotient is in AX
       MOV BL, 5
       DIV BL           ; 16 bit divide, quotient is in AL, r in AH
       XOR AH, AH       ; clear remainder
       MOV TWO_US, AX   ; store count for 0.2 us for ref.

       CALL TRYCY
       POP BP, DX, CX, BX, AX

       RET


; This cycle counting loop is intended to be about the same length as
; the core counting loop of HOWFAST.  Enter with number of iterations in
; CX

CYCOUNT:
      AND CX, CX ; Is CX = 0?
      JE > L4
L1:   CMP CRC, 0 ; I do need a memory read ...
      LOOP L1
L4:   RET

; A trial 10 second run of CYCOUNT.  Ten seconds requires 50 million calls
; to the 0.2 us routine, 50M = 2FAF080

STRY10: DB 'Start timing 10 sec$'
STRY1A: DB 'Stop timing 10 sec$'

TRYCY:
        CURSET 01400H ;; Row 20, col 0
        PRINTL STRY10
        MOV DX, 02FAh ; high count
        MOV AX, 0F080h ; low of 50,000,000
        MOV BX, TWO_US ; STORE COUNT IN BX
L1:     MOV CX, BX
;        CALL CYCOUNT

; start counting loop:

L9:     CMP CRC, 0 ; do a memory read ...
L8:     DEC CX
        JNZ L9

; end counting loop for 0.2uS

        DEC AX
        JNZ L1
        AND DX, DX ; If high and low both zero, quit
        JZ > L2
        DEC DX
        JMP L1
L2:     CURSET 01400H
        PRINTL STRY1A
        RET

ENDOFCOMMENT\


COMMENT\

VERIFIED TIMER TICK WAS ACCURATE WITH THIS ROUTINE
TRYCY:
        CURSET 01400h
        PRINTL STRY10
        MOV VARBL, 182
L1:     CMP VARBL, 0
        JNE L1
        CURSET 01400H
        PRINTL STRY1A
        RET
ENDOFCOMMENT\




;               TIMING ADJUST SUBROUTINE

; This subroutine lets the user input his own values for
; various serial bus timing loops. This is in hopes of curing
; 'ole Joe Woyak's timing problem once and for all. The calling
; program has cleared the screen and will restore it on return.

TIMING:
L2:	CALL SCURRENT; Put current values into strings
	CALL CLR_SCRN
        PRINTL TMENU
        CALL GET_TORK
	CALL TO_UPPER; 3.20
L1:     CMP AL, 'G' ; 4.09 - diagnostic function
        JNE > P2
        CALL DIAGS
        JMP > L7
P2:     CMP AL, 'D'
        JNE > L3
        CALL D1050
        JMP > L7
L3:     CMP AL, 'C' ; Change disk config: rev. 3.01
        JNE > L1
AEC:    CALL DISK_CONFG
	JMP L2; 3.20
L1:     CMP AL, 'T'   ; rev 3.06 addition
	JNE >L1
L6:	CALL T_RI_SENSE
	JMP L2
L1:     SUB AL,'1'
        CMP AL,9; REV 2.10, WAS 8 BEFORE
        JB >L1
	JMP >L7 ; Quit if other than 1 thru 8
L1:     XOR AH,AH
        MOV DI,AX
        SHL DI,1
        PUSH DI; DI now points to word 0 - 8

        PRINTL ENTER4
        CALL GET4H; Hex number is returned in BX
        POP DI
	JC >L7 ; Carry means ESC pressed
        MOV [T1+DI],BX; Good #, so put into timing database
	JMP L2

L7:    RET


DISK_CONFG:
	PRINTL CUR_DSTS
	MOV DI, 0
	MOV CX, 4; 4 DISKS TO CHECK

A2:     PUSH CX
        CMP B[WRITTN+DI],' '; Does this disk exist?
        JE > A1
        LEA DX, [S_DISK + DI]
	WRITE_SCREEN 8
	MOV AL, 'Y'
        TEST B[DSK_FLAGS+DI],2; B1 is "NO CONFG"
        JE >L1
	MOV AL,'N'
L1:     CALL PRINT1; one byte to screen
	CALL LINE_FEED
A1:     POP CX
	ADD DI, BLOCK_SIZE
        LOOP A2
	PRINTL ENT_NCH
	CALL GET_TORK
	CALL SET_BP; Carry clear if disk exists
        JC >L1
        XOR B[DSK_FLAGS+BP],2; toggle configurability status
	CALL LINE_FEED
	JMP DISK_CONFG

L1:     RET


; This subroutine takes a hex # in AX and puts the 4 ASCII hex
; digits to the field "HEX_PLACE"

HEX_IT:
        HEXTEX HEX_PLACE
        RET

; This routine takes the 16 bit numbers found in T1 thru T7,
; converts them to ASCII equivalents, and puts the ASCII strings
; into the strings ST1 thru ST9, getting them ready for printing.

SCURRENT:
        MOV CX,8; counter, 8 items (BEFORE 2.10 WAS 7)
L1:     MOV BX,CX; BX will be index to specific WORD value
        DEC BX; do for 0 thru 6
        SHL BX,1; Multiply by 2 to point to WORD
        MOV AX,[T1+BX]; Get actual word in AX
        CALL HEX_IT; Convert to string @ HEX_PLACE
        PUSH CX
        MOV DI,[STROFS+BX]; Get string offset pointer into DI
        MOV CX,4; Number of bytes to move
        PUSH CS
        POP ES
        MOV SI,OFFSET HEX_PLACE
        CLD
        REP MOVSB

        POP CX
        LOOP L1
        RET


; A routine to program timer 0 to what it should already be:
; mode 3, count = 0 (2^16)
; Rev 3.18 change: was setting to MODE 2, now MODE 3 ...

PRG_T0:
        MOV DX,043h; Address control port
	MOV AL,00110110xB; timer 0, 16 bit, mode 3, binary 3.18
        OUT DX,AL
        MOV AL,0; now, output the count (0000)
        MOV DX,040h; Timer 0 port
        OUT DX,AL
        CALL TIME_LAG
        OUT DX,AL
        RET



; A routine to delay the number of 420 ns cycles as are stored
; in WORD @ T7. Used for delay between accesses to UART

ACC_DLY:
        CMP T7,0 ; Just return if zero
        JE >L1
        PUSH BX
        MOV BX,T7
        CALL TIMER_0
        POP BX
L1:     RET


;       TIMER_0

; rev 4.10 is converting TIMER_0 to actually use timer #2 ...
;
; A routine which waits until timer 0 has decremented the number
; of counts found in register BX. Each count takes about 420 nSec.
; Note: don't try this with BX = 0FFFFh; the CX < BX check is bound
; to always be true.  In fact, I recommend that you don't even get
; close to 0FFFF, say within 1000, unless your computer is pretty
; fast, because "rollunder" will cause you to miss the event.
; Note that prior to 3.18, I thought each count was 840 nSec, but I've
; found out that the counter is counting by TWOS!

; 3.18 Changed this routine so it always set the timer up with a count
; of 0 and restarted.  For some reason, this caused problems with PUTSEC
; (SIO2PC receiving sectors from the Atari).  So 3.19 restores the old
; TIMER_0 method.

; Rev. 4.20 is moving timer_0 to a routine (below) that uses mode 0
; and reads the out pin ...

; Attempt to use one-shot mode of timer 8254 by reading the OUT pin
; status

COMMENT\ I'm commenting this one out (although it works), because I want
; to go back to timer 0 and overcome my fear of rollunder.  (And also
; allow sound via the speaker/timer-2/port-61)
; so here we go again (still in rev 4.20), the real timer_0 is below

TIMER_0:
        PUSH AX, DX
        SHR BX, 1

; (above) divide BX by 2 because 8254 counts by 1's in mode 0, in mode
; 3, it counted by twos, so my number of counts was set for that ...
; 4.20 change ...

        MOV DX, 061h
        IN AL, DX
        MOV W[COUNT_TIME], 0
        AND AL, 0FFh - 3; turn off bits 0 and 1
        JMP > L1
L1:     OUT DX, AL
        JMP > L7
L7:     MOV DX, 043h ; control word's address
        MOV AL, 10110000xB ; counter 2, mode 0; write LSB, then MSB
        OUT DX, AL
        JMP > L2
L2:     MOV AL, BL ; I'll use decimal count of 100 for test
        MOV DX, 042h ; counter 2's address
        OUT DX, AL ; put LSB of count
        JMP > L3
L3:     MOV AL, BH
        OUT DX, AL ; put MSB of count
        JMP > L1
L1:     MOV DX, 061h ; time to turn gating on ...
        IN AL, DX
        OR AL, 1 ; by setting bit 0
        JMP > L2
L2:     OUT DX, AL ; start the counter

; I'm going to verify the null count flag before looking for the OUT pin's
; status

S1:     INC W[COUNT_TIME]
        MOV DX, 043h
        MOV AL, 11101000xB ; Read back command: status of counter #2
        JMP > L3
L3:     OUT DX, AL
        MOV DX, 042h ; address counter 2 to read its status byte
        JMP > L1
L1:     IN AL, DX ; get counter 2's status byte to AL
        AND AL, 01000000xB ; bit 6 is null count
        JNZ S1 ; if null is not 0, then not ready ... so loop back
        MOV W[COUNT_TIME], 0

; At this point, counter is running so I can watch the OUT pin ...

S2:     INC W[COUNT_TIME]
        MOV DX, 043h ; latch status same way as above
        MOV AL, 11101000xB ;
        JMP > L3
L3:     OUT DX, AL
        MOV DX, 042h ;
        JMP > L1
L1:     IN AL, DX ; status byte again, this time interested in b7
        AND AL, 10000000xB ; bit 7 is low during counting, then high
        JZ S2 ; so loop back if b7 is low

; Now, we're finished; OUT is high

       NOP
        POP DX, AX
        RET

ENDOFCOMMENT\


; I'm coming back to where I was originally with timer_0: reading timer 0
; in a loop and calculating the number of counts to pass.
; I've had to get past my irrational fears that the rollunder (past 0) is
; going to cause some kind of errors or maybe an infinite or near-infinite
; loop.  In fact, using ordinary non-signed math gives me an accurate count
; whether or not rollunder has occurred.
;
; The method is to recorded the initial latched count, then keep getting the
; current count and subtracting it from the current.  (Remember the counter
; counts DOWN.) The result is the number of counts.  Compare that to the
; desired counts and loop back if smaller than desired.
;
; Counter 0 is a good one to use because it's already programmed by the bios
; in mode 3 and runs with a full count, so there are no missing numbers as
; it goes from 0FFFFh to 0 and repeats.  Also please recall that in mode 3
; the counter is counting by two's.  So expect 420 nS per count, not 840.
;
; Using counter 0 also frees up counter 2 for sound or whatever.
;
; Note that there's no need to read the null count flag or anything because
; we aren't programming a mode or loading a count--we're just latching and
; reading counts.  OK?  NRK 1/21/99 1:40 AM

TIMER_0:

; BX has the # of clocks we want ...

        PUSH AX, CX, DX
        MOV DX, 043h ; ready to send the counter latch command
        MOV AL, 00000000xB ; latch counter # 0
        OUT DX, AL

; First, get the initial count:

        MOV DX, 040h ; to read counter # 0
        IN AL, DX ; get low byte of initial count
        MOV CL, AL ; store in CX
        MOV DX, 061h ; I'm going to kill 1 I/O cycle by reading port 60
        IN AL, DX
        MOV DX, 040h
        IN AL, DX ; now get high byte of count
        MOV CH, AL ; count is now in CX
        MOV INITIAL, CX

; Second, check the current count:


CKCUR:  MOV DX, 043h ; ready to send the counter latch command
        MOV AL, 00000000xB ; latch counter # 0
        OUT DX, AL
        MOV DX, 040h
        IN AL, DX
        MOV CL, AL
        MOV DX, 061h
        IN AL, DX
        MOV DX, 040h
        IN AL, DX
        MOV CH, AL

; Get the difference between current and initial

        MOV DX, INITIAL
        SUB DX, CX ; calculate # of counts passed; held in DX ...
        CMP DX, BX
        JB CKCUR ; read another count if difference is < target

; Now we're done, so save the results

        MOV DIFFERENCE, DX
        MOV FINAL, CX ; current has become final ...
        POP DX, CX, AX
        RET



;               DO_TIME

; This is a crude timer used to guard against getting stuck in a loop.
; Typically, at entry into the loop, you will put 0 into the word
; variable COUNT_TIME. Then, each time through, call this routine. If
; it returns with zero flag true, break out of the loop. Also, at entry
; into the loop, put the loop error ID character to variable B[ERR_CHAR].
; If this routine times out, it will put the character to field 
; STAT_EROR on the DEBUG status line.
; The time it takes is 65000*cycle time*# of cycles and is at least a
; few milliseconds, and could be several seconds in a big loop with a
; slow clock.

COMMENT\
THIS ROUTINE HAS CAUSED ME PROBLEMS WITH FAST COMPUTERS
(PENTIUMS), SO I'M GETTING RID OF IT!!!
DO_TIME:
        DEC W[COUNT_TIME]; yes, DEC does condition ZF.
        JNZ >L1
        CALL PUT_ERCH           
        CMP AX,AX; Make sure ZF is still set
L1:     RET
ENDOFCOMMENT\

; BEEP gives a beep over the console speaker

BEEP:
        CMP SOUND, 0
        JE > L3
        PUSH DX, AX
        MOV DX, 061h     ; TTL control port
        IN AL, DX

; Now, program counter 2 for speaker.

	AND AL,0FEh; Set off bit 0
	OUT DX,AL
	MOV DX,043h
        MOV AL,10110110xb; #2, 16 bit count, mode 3, binary
        OUT DX, AL
        MOV DX, 042h; Address counter #2
        MOV AX, BEEPCNT  ; 16 bit count
        OUT DX, AL
        XCHG AL, AH
        OUT DX, AL
        MOV DX, 061h; Now, turn on the gate
        IN AL, DX
        OR AL, 3; bits 0 and 1 on
        OUT DX, AL ; Hear anything?
        TICKS BEEPTIME
	MOV DX,061h
        IN AL, DX
        AND AL,0FFh - 2 ; Turn it off!!!!! (speaker output)
        OR AL, 1 ; Turn it on!! (gating input to counter #2)
        OUT DX, AL
        POP AX, DX
L3:     RET


; SPKR_1 energizes the speaker coil (assuming OUT2 of the 8254 is HIGH)
; This will cause a click if it was initially not energized

SPKR_1:
	CMP SOUND, 0
	JZ > L2
        PUSH AX, DX
        MOV DX, 061h ; bit 1 of this port controls the speaker
	IN AL, DX    ; (ANDed with the OUT2 of the timer chip)
        OR AL, 2     ; set bit 1 to 1
	JMP > L1     ; kill some time for slow I/O
L1:	OUT DX, AL
        POP DX, AX
L2:	RET

; SPKR_0 de-energizes the speaker
; This will cause a click if it was initially energized


SPKR_0:
	CMP SOUND, 0
	JZ > L2
SPOFF: PUSH AX, DX ; SPOFF is entry point for absolute "turn it off"
	MOV DX, 061h
	IN AL, DX
        AND AL, 0FFh - 2 ; clear bit 1 to 0
	JMP > L1
L1:	OUT DX, AL
        POP DX, AX
L2:	RET


; SNDONOFF turns the sound effects ON and OFF

SNDONOFF:
        CMP SOUND, 0 ; sound OFF??
        JE > L1
        PSTATUS SOUNDOFF, ATTR9
        MOV SOUND, 0
        CALL SPOFF ; make sure speaker is off
        JMP > L2
L1:     MOV SOUND, 0FFh
        PSTATUS SOUNDON, ATTR9
L2:     TICKS 22
        RET

